#!/bin/bash

# ====================================
# Installation certificats RAMQ - Mac
# Version avec téléchargement depuis serveur Info-Data
# ====================================

# Couleurs pour l'affichage
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
CYAN='\033[0;36m'
GRAY='\033[0;90m'
NC='\033[0m'

echo ""
echo "===================================="
echo "Installation certificats RAMQ"
echo "Script Mac"
echo "===================================="
echo ""

# ====================================
# Vérification des privilèges root
# ====================================

if [ "$EUID" -ne 0 ]; then 
    echo -e "${RED}✗ Ce script doit être exécuté avec sudo${NC}"
    echo ""
    echo "Utilisez: sudo bash install-ramq-mac.sh"
    echo ""
    exit 1
fi

# ====================================
# Détection automatique de Java
# ====================================

echo -e "${YELLOW}2. Détection de Java...${NC}"

JAVA_HOME=""

# Méthode 1 : Variable d'environnement JAVA_HOME
if [ -n "$JAVA_HOME" ] && [ -f "$JAVA_HOME/bin/keytool" ]; then
    echo -e "   ${GREEN}✓ Java trouvé via JAVA_HOME${NC}"
# Méthode 2 : Utiliser /usr/libexec/java_home (spécifique Mac)
elif command -v /usr/libexec/java_home &> /dev/null; then
    JAVA_HOME=$(/usr/libexec/java_home 2>/dev/null)
    if [ -n "$JAVA_HOME" ] && [ -f "$JAVA_HOME/bin/keytool" ]; then
        echo -e "   ${GREEN}✓ Java trouvé: $JAVA_HOME${NC}"
    fi
# Méthode 3 : Chercher java dans le PATH
elif command -v java &> /dev/null; then
    JAVA_PATH=$(which java)
    while [ -L "$JAVA_PATH" ]; do
        JAVA_PATH=$(readlink "$JAVA_PATH")
    done
    JAVA_HOME=$(cd "$(dirname "$JAVA_PATH")/.." && pwd)
    echo -e "   ${GREEN}✓ Java trouvé via PATH${NC}"
# Méthode 4 : Chercher dans les emplacements standards
else
    SEARCH_PATHS=(
        "/Library/Java/JavaVirtualMachines/*/Contents/Home"
        "/System/Library/Java/JavaVirtualMachines/*/Contents/Home"
        "/usr/local/opt/openjdk*/libexec/openjdk.jdk/Contents/Home"
        "/Library/Internet Plug-Ins/JavaAppletPlugin.plugin/Contents/Home"
    )
    
    for SEARCH_PATH in "${SEARCH_PATHS[@]}"; do
        for DIR in $SEARCH_PATH; do
            if [ -f "$DIR/bin/keytool" ]; then
                JAVA_HOME="$DIR"
                echo -e "   ${GREEN}✓ Java trouvé: $JAVA_HOME${NC}"
                break 2
            fi
        done
    done
fi

# Vérification finale
if [ -z "$JAVA_HOME" ] || [ ! -f "$JAVA_HOME/bin/keytool" ]; then
    echo -e "${RED}✗ Java non trouvé sur ce système!${NC}"
    echo ""
    echo -e "${YELLOW}Solutions:${NC}"
    echo "1. Installez Java (JDK ou JRE)"
    echo "2. Définissez la variable JAVA_HOME"
    echo "3. Installez Java via Homebrew: brew install openjdk"
    echo ""
    exit 1
fi

KEYTOOL="$JAVA_HOME/bin/keytool"
CACERTS="$JAVA_HOME/lib/security/cacerts"

# Pour les anciennes versions de Java
if [ ! -f "$CACERTS" ]; then
    CACERTS="$JAVA_HOME/jre/lib/security/cacerts"
fi

if [ ! -f "$CACERTS" ]; then
    echo -e "${RED}✗ Truststore non trouvé!${NC}"
    exit 1
fi

echo -e "   ${GREEN}✓ Truststore: $CACERTS${NC}"
echo ""

# ====================================
# Chargement des certificats
# ====================================

echo -e "${YELLOW}1. Chargement des certificats RAMQ...${NC}"

# Dossier des certificats (relatif au script)
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
CERT_FOLDER="$SCRIPT_DIR/cert_ramq"

# Vérifier que le dossier existe
if [ ! -d "$CERT_FOLDER" ]; then
    echo -e "${RED}✗ Dossier cert_ramq introuvable!${NC}"
    echo ""
    echo -e "${YELLOW}Le dossier cert_ramq doit se trouver à côté du script.${NC}"
    echo -e "${YELLOW}Vérifiez que vous avez bien extrait tout le contenu du ZIP.${NC}"
    echo ""
    exit 1
fi

TEMP_DIR=$(mktemp -d)
CERT_FILES=()
CERT_INDEX=0

for i in 0 1 2; do
    CERT_PATH="$CERT_FOLDER/ramq-cert-$i.cer"
    
    if [ -f "$CERT_PATH" ]; then
        # Copier dans le dossier temporaire
        TEMP_CERT="$TEMP_DIR/ramq-cert-$i.cer"
        cp "$CERT_PATH" "$TEMP_CERT"
        
        # Afficher le nom du certificat (optionnel, ne bloque pas si OpenSSL manque)
        if command -v openssl &> /dev/null; then
            SUBJECT=$(openssl x509 -in "$TEMP_CERT" -noout -subject 2>/dev/null | sed 's/subject=//')
            if [ -n "$SUBJECT" ]; then
                echo -e "   ${GREEN}✓ Certificat $i : $SUBJECT${NC}"
            else
                echo -e "   ${GREEN}✓ Certificat $i chargé${NC}"
            fi
        else
            echo -e "   ${GREEN}✓ Certificat $i chargé${NC}"
        fi
        
        CERT_FILES+=("$TEMP_CERT")
    else
        echo -e "   ${YELLOW}⚠ Certificat $i manquant : $CERT_PATH${NC}"
    fi
    
    CERT_INDEX=$((CERT_INDEX + 1))
done

if [ ${#CERT_FILES[@]} -eq 0 ]; then
    echo -e "${RED}✗ Aucun certificat trouvé${NC}"
    echo ""
    echo -e "${YELLOW}Vérifiez que le dossier cert_ramq contient les fichiers :${NC}"
    echo "  - ramq-cert-0.cer"
    echo "  - ramq-cert-1.cer"
    echo "  - ramq-cert-2.cer"
    echo ""
    rm -rf "$TEMP_DIR"
    exit 1
fi

echo ""
echo -e "   ${GREEN}✓ ${#CERT_FILES[@]} certificat(s) chargé(s)${NC}"
echo ""

# ====================================
# Installation des certificats
# ====================================

echo -e "${YELLOW}3. Installation des certificats dans Java...${NC}"

INSTALLED_COUNT=0
CERT_INDEX=0

for CERT_FILE in "${CERT_FILES[@]}"; do
    ALIAS="ramq-cert-$CERT_INDEX"
    
    # Supprimer l'ancien si existe
    "$KEYTOOL" -delete -alias "$ALIAS" -keystore "$CACERTS" -storepass changeit 2>/dev/null
    
    # Installer le certificat
    "$KEYTOOL" -import -alias "$ALIAS" -keystore "$CACERTS" -file "$CERT_FILE" -storepass changeit -noprompt >/dev/null 2>&1
    
    if [ $? -eq 0 ]; then
        echo -e "   ${GREEN}✓ Installé: $ALIAS${NC}"
        INSTALLED_COUNT=$((INSTALLED_COUNT + 1))
    else
        echo -e "   ${RED}✗ Erreur: $ALIAS${NC}"
    fi
    
    CERT_INDEX=$((CERT_INDEX + 1))
done

echo ""

# ====================================
# Vérification
# ====================================

echo -e "${YELLOW}4. Vérification de l'installation...${NC}"

ALL_OK=true
CERT_INDEX=0

for CERT_FILE in "${CERT_FILES[@]}"; do
    ALIAS="ramq-cert-$CERT_INDEX"
    
    "$KEYTOOL" -list -alias "$ALIAS" -keystore "$CACERTS" -storepass changeit >/dev/null 2>&1
    
    if [ $? -eq 0 ]; then
        echo -e "   ${GREEN}✓ $ALIAS présent${NC}"
    else
        echo -e "   ${RED}✗ $ALIAS manquant${NC}"
        ALL_OK=false
    fi
    
    CERT_INDEX=$((CERT_INDEX + 1))
done

echo ""

# ====================================
# Nettoyage des processus et caches
# ====================================

echo -e "${YELLOW}5. Nettoyage des processus et caches...${NC}"

# Arrêter les processus Java
JAVA_PIDS=$(pgrep -f "java.*medoffice" 2>/dev/null)

if [ -n "$JAVA_PIDS" ]; then
    echo -e "   ${GRAY}Arrêt des processus Java...${NC}"
    echo "$JAVA_PIDS" | while read PID; do
        echo -e "     ${GRAY}- Arrêt du processus $PID${NC}"
        kill -9 "$PID" 2>/dev/null
    done
    sleep 2
    echo -e "   ${GREEN}✓ Processus Java arrêtés${NC}"
else
    echo -e "   ${CYAN}ℹ Aucun processus Java en cours${NC}"
fi

# Nettoyer les caches medoffice
MEDOFFICE_PATH="$HOME/medoffice"

if [ -d "$MEDOFFICE_PATH" ]; then
    echo -e "   ${GRAY}Nettoyage des caches medoffice...${NC}"
    
    if [ -d "$MEDOFFICE_PATH/configuration" ]; then
        rm -rf "$MEDOFFICE_PATH/configuration/"* 2>/dev/null
        echo -e "   ${GREEN}✓ Cache configuration supprimé${NC}"
    fi
    
    if [ -d "$MEDOFFICE_PATH/instance/.metadata/.plugins" ]; then
        rm -rf "$MEDOFFICE_PATH/instance/.metadata/.plugins/"* 2>/dev/null
        echo -e "   ${GREEN}✓ Cache instance supprimé${NC}"
    fi
else
    echo -e "   ${CYAN}ℹ Dossier medoffice non trouvé${NC}"
fi

echo ""

# ====================================
# Nettoyage des fichiers temporaires
# ====================================

rm -rf "$TEMP_DIR"

# ====================================
# Résumé final
# ====================================

echo "===================================="
if [ "$ALL_OK" = true ] && [ $INSTALLED_COUNT -gt 0 ]; then
    echo -e "${GREEN}✓ Installation RÉUSSIE!${NC}"
    echo -e "${GREEN}$INSTALLED_COUNT certificat(s) installé(s)${NC}"
else
    echo -e "${YELLOW}⚠ Installation PARTIELLE${NC}"
    echo -e "${YELLOW}$INSTALLED_COUNT/${#CERT_FILES[@]} certificat(s) installé(s)${NC}"
fi
echo "===================================="
echo ""
echo -e "${YELLOW}PROCHAINES ÉTAPES:${NC}"
echo "1. Relancez votre application Java"
echo "2. Testez la connexion à la RAMQ"
echo ""
echo -e "${YELLOW}Si le problème persiste:${NC}"
echo "- Vérifiez que l'application utilise ce Java: $JAVA_HOME"
echo "- Contactez le support technique"
echo ""
